//  -*- Mode: C++;  -*-

//  File: mkgrid.cpp

//  Author: Dino Bellugi (dino@geomorph.berkeley.edu)

//  Copyright  Dino Bellugi, BlueG SoftWear, U.C. Berkeley, 1999 (C)

// *~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

// * FUNCTION: mkgrid, creates grid files for tests and simulations

// *

// * RELATED PACKAGES: grid.cpp

// *

// * HISTORY:

// * Created:  Thu Nov  2 1995 (dino)

// * Modified: Sat Jun 26 1999 (dino)

// *~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~



// the following include file (notice.h) contains important copyright 

// information, as well as other legal statements. it must be present 

// in all files and in any distribution. removal or modification is

// strictly forbidden and violates u.s. and international copyright law.

#include "notice.h"

char *copyright_notice = "(C) 1999 Dino Bellugi, BlueG SoftWear, U.C. Berkeley";

//

#include <stdio.h>

#include <stdlib.h>

#include <fstream.h>

#include <string.h>

#include <math.h>

#include "grid.h"

#include "macros.h"



main(int argc, char **argv) 

{

	Grid grd;

	int i, j, k, m, n, nrows, ncols, type, mode;

	double size, x, y, val1, val2, val3, val4;



	// Get the arguments from the command line.

	if (argc < 6) 

	{

    		// Problem With input

    		cerr << "usage:\n\t" << appName(argv) <<

			" <format> <rows> <columns> <cell size> <mode> [<values>]\n";

                cerr << "formats:\n";

                cerr << "\t0 = arcinfo ascii (default)\n";

                cerr << "\t1 = surfer ascii\n";

		cerr << "modes:\n";

		cerr << "\t0 - nodata grid\n";

		cerr << "\t1 - constant <value> grid\n";

		cerr << "\t2 - <value1> grid, 1 in <value2> prob. of <value3>, <value4> (odd) box\n";

		cerr << "\t3 - y = x * <value1> + <value2>\n";

		cerr << "\t4 - y = x * x * <value1> + <value2>\n";

		cerr << "\t5 - y = x * x * <value1> + x * <value2> + <value3>\n";

		cerr << "\t6 - z = (x^2 + y^2)^0.5 * <value1> + <value2>\n";

		cerr << "\t7 - z = (x^2 + y^2) * <value1> + <value2>\n";

		cerr << "\t8 - z = (x^2 + y^2) * <value1> + (x^2 + y^2)^0.5 * <value2> + <value3>\n";

		cerr << "\t9 - text file <file name>\n";

                cerr << "example:\n\t";

                cerr << appName(argv) << "0 10 10 5 1 0 > zero.grd\n";



   		exit(1);

  	}



        // Get format

        if ((type = atoi(argv[1])) <= 0 || type > 1)

                type = ARCINFO;

        else

                type = SURFER;

	nrows = atoi(argv[2]);

	ncols = atoi(argv[3]);

	size  = atof(argv[4]);

	mode  = atoi(argv[5]);



	cerr << "Initializing output grid ...\n";



	switch (type)

	{

		case 0:

			if (!grd.initGrid(nrows, ncols, TRUE, size))

				exit(1);

			break;



		case 1:

			val1 = atof(argv[6]);

			if (!grd.initGrid(nrows, ncols, TRUE, size, val1))

				exit(1);

			break;



		case 2:

		{

			unsigned long lRn, lPr;



			val1 = atof(argv[6]);

			val2 = atof(argv[7]);

			val3 = atof(argv[8]);

			val4 = atof(argv[9]);

			m = trn(val4 / 2);



			if (!grd.initGrid(nrows, ncols, TRUE, size, val1))

				exit(1);



			// do random here. 

			// probability is val2 / val4^2

			// that the box around cell is val3



			// compute probability

			lPr = ulrnd(val2 * sqr(val4));



			for (j = 0; j < grd.ydim; ++j)

			{

				for (i = 0; i < grd.xdim; ++i)

				{

					// pick a random number

					lRn = grd.getRandLong(sqr(lPr));



					// is it in range?

					if (lRn < lPr)

					{

						grd.at(i, j) = val3;

						for (n = 1; n <= m; ++n)

						{

							grd.at(i - n, j)     = val3;

							grd.at(i, j - n)     = val3;

							grd.at(i - n, j - n) = val3;

							grd.at(i + n, j)     = val3;

							grd.at(i, j + n)     = val3;

							grd.at(i + n, j + n) = val3;

							grd.at(i - n, j + n) = val3;

							grd.at(i + n, j - n) = val3;

							for (k = 1; k < n; ++k)

							{

								grd.at(i - n, j - k)     = val3;

								grd.at(i + k, j - n)     = val3;

								grd.at(i - n + k, j - n) = val3;

								grd.at(i + n, j + k)     = val3;

								grd.at(i - k, j + n)     = val3;

								grd.at(i + n - k, j + n) = val3;

								grd.at(i - n, j + n - k) = val3;

								grd.at(i + n, j - n + k) = val3;

							}

						}

					}

				}

			}  

			grd.setNodata(DEF_VAL);

			break;

		}



		case 3:

			val1 = atof(argv[6]);

			val2 = atof(argv[7]);

			if (!grd.initGrid(nrows, ncols, TRUE, size))

				exit(1);

			for (j = 0; j < grd.ydim; ++j)

			{

				for (i = 0, n = grd.xdim / -2; i < grd.xdim; ++i, ++n)

				{

					x = n * size;

					grd.at(i, j) = x * val1 + val2;

				}

			}  

			break;



		case 4:

			val1 = atof(argv[6]);

			val2 = atof(argv[7]);

			if (!grd.initGrid(nrows, ncols, TRUE, size))

				exit(1);

			for (j = 0; j < grd.ydim; ++j)

			{

				for (i = 0, n = grd.xdim / -2; i < grd.xdim; ++i, ++n)

				{

					x = n * size;

					grd.at(i, j) = sqr(x) * val1 + val2;

				}

			}  

			break;



		case 5:

			val1 = atof(argv[6]);

			val2 = atof(argv[7]);

			val3 = atof(argv[8]);

			if (!grd.initGrid(nrows, ncols, TRUE, size))

				exit(1);

			for (j = 0; j < grd.ydim; ++j)

			{

				for (i = 0, n = grd.xdim / -2; i < grd.xdim; ++i, ++n)

				{

					x = n * size;

					grd.at(i, j) = sqr(x) * val1 + x * val2 + val3;

				}

			}  

			break;



		case 6:

			val1 = atof(argv[6]);

			val2 = atof(argv[7]);

			if (!grd.initGrid(nrows, ncols, TRUE, size))

				exit(1);

			for (j = 0, n = grd.ydim / -2; j < grd.ydim; ++j, ++n)

			{

				y = n * size;

				for (i = 0, m = grd.xdim / -2; i < grd.xdim; ++i, ++m)

				{

					x = m * size;

					grd.at(i, j) = sqrt(sqr(x) + sqr(y)) 

						* val1 + val2;

				}

			}  

			break;



		case 7:

			val1 = atof(argv[6]);

			val2 = atof(argv[7]);

			if (!grd.initGrid(nrows, ncols, TRUE, size))

				exit(1);

			for (j = 0, n = grd.ydim / -2; j < grd.ydim; ++j, ++n)

			{

				y = n * size;

				for (i = 0, m = grd.xdim / -2; i < grd.xdim; ++i, ++m)

				{

					x = m * size;

					grd.at(i, j) = (sqr(x) + sqr(y)) 

						* val1 + val2;

				}

			}  

			break;



		case 8:

			val1 = atof(argv[6]);

			val2 = atof(argv[7]);

			val3 = atof(argv[8]);

			if (!grd.initGrid(nrows, ncols, TRUE, size))

				exit(1);

			for (j = 0, n = grd.ydim / -2; j < grd.ydim; ++j, ++n)

			{

				y = n * size;

				for (i = 0, m = grd.xdim / -2; i < grd.xdim; ++i, ++m)

				{

					x = m * size;

					grd.at(i, j) = (sqr(x) + sqr(y)) *

						val1 + sqrt(sqr(x) + sqr(y)) 

						* val2 + val3;

				}

			}  

			break;



		case 9:

		{

			ifstream infile(argv[6]);

			if(!infile) 

			{

				cerr << appName(argv) <<

					": unable to find file " <<

					argv[6] << "\n";

				exit(1);

			}

			infile.precision(20);

			infile.width(20);



			if (!grd.initGrid(nrows, ncols, TRUE, size))

				exit(1);



			for (j = 0; j < grd.ydim; ++j)

				for (i = 0; i < grd.xdim; ++i)

					infile >> grd.at(i, j);

			break;

		}



		default:

			cerr << appName(argv) << ": mode " << mode <<

				" not implemented ...\n";

			exit(1);

	}



	cerr << "Writing output grid ...\n";

	grd.coutGrid(0, type);



        exit(FALSE);    

        return(TRUE);

}

